import java.util.Iterator;

//-----------------------------------------------
// Recursive implementation of BinaryTree matching Carrano&Pritchard interface
//
// Chosen representation for an empty tree:
// leftTree == rightTree == null  (allows null items to be stored)
//
// Implementation properties:
// leftTree  == null => this is empty 
// rightTree == null => this is empty 
// parent    == null => this is not a subtree of some other tree
//    (required to guarantee trees remain acyclic and no node has two parents)
//
// written by Frank Tompa for CS 134, May 2004
//    with advice from all W04 and S04 CS 134 instructors
 //     toString added by Paul Nijjar, July 5 2004
//      updated by Brian Laframboise, Dec 2004
//-----------------------------------------------

public class BinaryTree implements BinaryTreeInterface
{
  private Object item;
  private BinaryTree leftTree;
  private BinaryTree rightTree;
  private BinaryTree parent;

  public BinaryTree()
  //post: initializes a new empty BinaryTree
  {
    setParent(null);
    makeEmpty();
  }

  public BinaryTree(Object rootItem)
  // post: initializes a new BinaryTree that contains rootItem
  //       and has empty left and right subtrees
  {
    setParent(null);
    makeEmpty();
    setRootItem(rootItem);
  }

  public BinaryTree(Object rootItem, BinaryTreeInterface newLeft,
                      BinaryTreeInterface newRight)
  // pre:  newLeft != null, !newLeft.isEmpty()
  //       newRight != null, !newRight.isEmpty()
  // post: initalizes a new BinaryTree that contains rootItem
  //       and attaches newLeft and newRight as its left and right subtrees
  //       newLeft and newRight are empty
  {
    setParent(null);
    setRootItem(rootItem);
    attachLeftSubtree(newLeft);
    attachRightSubtree(newRight);
  }

  //----------------------------------
  // NestingInterface implementations
  //----------------------------------

  public void makeEmpty()
  // Removes all nodes from the tree.
  // post: isEmpty()
  {
    item = null;
    leftTree = null;
    rightTree = null;
  }

  public boolean isEmpty()
  // Determines whether or not the tree is empty
  // post: returns true iff the tree is empty
  {
    // would work equally well to check rightTree
    return leftTree == null;
  }

  public Object getRootItem() throws TreeException
  // Retrieves the value in the root node.
  // pre:  !isEmpty()
  // post: returns the value stored in the root node
  //       throws a TreeException if the tree is empty
  {
    if (isEmpty())
      throw new TreeException("TreeException: Empty tree");

    return item;
  }

  public void setRootItem(Object newItem)
  // Sets the value of the root node.
  // post: the value of the root is newItem
  //       the tree has left and right subtrees
  {
    if (isEmpty())
    {
      leftTree = cloneEmptyTree();
      leftTree.setParent(this);
      rightTree = cloneEmptyTree();
      rightTree.setParent(this);
    }
    item = newItem;
  }

  //-------------------------------------
  // BinaryTreeInterface implementations 
  //-------------------------------------

  public void attachLeft(Object newItem)
  // Appends a node as the left subtree
  // pre:  !isEmpty()
  // post: If this has an empty left subtree,
  //       then it is replaced by a left subtree that contains newItem
  //       and has empty left and right subtrees.
  //       Otherwise the tree is unchanged.
  {
    if (!isEmpty() && hasEmptyLeft())
    {
      // assertion: nonempty tree; no left child
      leftTree.setRootItem(newItem);
    }
  }

  public void attachRight(Object newItem)
  // Appends a node as the right subtree
  // pre:  !isEmpty()
  // post: If this has an empty right subtree,
  //       then it is replaced by a right subtree that contains newItem
  //       and has empty left and right subtrees.
  //       Otherwise the tree is unchanged.
  {
    if (!isEmpty() && hasEmptyRight())
    {
      // assertion: nonempty tree; no right child
      rightTree.setRootItem(newItem);
    }
  }

  public void attachLeftSubtree(BinaryTreeInterface newTree)
                                                    throws TreeException
  // Appends a pre-formed tree as the left subtree
  // post: newTree is appended as the left subtree.
  //       newTree itself is made empty.
  //       throws an exception if:
  //       - this or newTree isEmpty()
  //       - this has a non-empty left subtree
  //       - this is nested within newTree
  //       - newTree is nested within some other tree
  {

    if (isEmpty())
    {
      throw new TreeException("TreeException: " +
        "Cannot attach a left subtree to an empty tree");
    }
    else if (newTree.isEmpty())
    {
      throw new TreeException("TreeException: " +
        "Cannot attach an empty left subtree");
    }
    else if (!hasEmptyLeft())
    {
      // a left subtree already exists; it should have been deleted first
      throw new TreeException("TreeException: " +
        "Cannot use attachLeftSubtree to overwrite left subtree");
    }
    else if (((BinaryTree)newTree).isNested())
    {
      // newTree is already the subtree of some other tree;
      // it should have been detached from there first
      throw new TreeException("TreeException: " +
        "Cannot attach a subtree of some other tree");
    }
    else if (newTree == rootAncestor())
    {
      // this is in a tree rooted at newTree
      throw new TreeException("TreeException: " +
        "Cannot attach a tree as a descendant subtree of itself");
    }

    // Assertion: this/newTree is non-empty, this has non-empty left,
    //            newTree is not nested, this is not rooted at newTree

    leftTree.setRootItem(newTree.getRootItem());

    // Recursively copy the left subtree into leftTree
    BinaryTreeInterface newLeft = newTree.detachLeftSubtree();

    if(!newLeft.isEmpty())
      leftTree.attachLeftSubtree(newLeft);

    // Recursively copy the right subtree into leftTree
    BinaryTreeInterface newRight = newTree.detachRightSubtree();

    if(!newRight.isEmpty())
      leftTree.attachRightSubtree(newRight);

    newTree.makeEmpty();

  } 

  public void attachRightSubtree(BinaryTreeInterface newTree)
                                          throws TreeException
  // Appends a pre-formed tree as the right subtree
  // post: newTree is appended as the right subtree.
  //       newTree itself is made empty.
  //       throws an exception if:
  //       - this or newTree isEmpty()
  //       - this has a non-empty right subtree
  //       - this is nested within newTree
  //       - newTree is nested within some other tree
  {
    if (isEmpty())
    {
      throw new TreeException("TreeException: " +
        "Cannot attach a right subtree to an empty tree");
    }
    else if (newTree.isEmpty())
    {
      throw new TreeException("TreeException: " +
        "Cannot attach an empty right subtree");
    }
    else if (!hasEmptyRight())
    {
      // a right subtree already exists; it should have been deleted first
      throw new TreeException("TreeException: " +
        "Cannot use attachRightSubtree to overwrite right subtree");
    }
    else if (((BinaryTree)newTree).isNested())
    {
      // newTree is already the subtree of some other tree;
      // it should have been detached from there first
      throw new TreeException("TreeException: " +
        "Cannot attach a subtree of some other tree");
    }
    else if (newTree == rootAncestor())
    {
      // this is in a tree rooted at newTree
      throw new TreeException("TreeException: " +
        "Cannot attach a tree as a descendant subtree of itself");
    }

    // Assertion: this/newTree is non-empty, this has non-empty right,
    //            newTree is not nested, this is not rooted at newTree

    rightTree.setRootItem(newTree.getRootItem());

    // Recursively copy the left subtree into rightTree
    BinaryTreeInterface newLeft = newTree.detachLeftSubtree();

    if(!newLeft.isEmpty())
      rightTree.attachLeftSubtree(newLeft);

    // Recursively copy the right subtree into rightTree
    BinaryTreeInterface newRight = newTree.detachRightSubtree();

    if(!newRight.isEmpty())
      rightTree.attachRightSubtree(newRight);

    newTree.makeEmpty();
  } 

  public BinaryTreeInterface detachLeftSubtree() throws TreeException
  // Detaches the left subtree
  // post: the left subtree is detached and returned, and
  //       the new left subtree of this is empty.
  //       throws an exception if this is empty
  {
    if (isEmpty())
      throw new TreeException("TreeException: Cannot detach left subtree of an empty tree");

    BinaryTree oldLeft = leftTree;
    oldLeft.setParent(null);

    leftTree = cloneEmptyTree();
    leftTree.setParent(this);

    return oldLeft;
  }

  public BinaryTreeInterface detachRightSubtree() throws TreeException
  // Detaches the right subtree
  // post: the right subtree is detached and returned, and
  //       the new right subtree of this is empty.
  //       throws an exception if isEmpty()
  {
    if (isEmpty())
      throw new TreeException("TreeException: Cannot detach right subtree of an empty tree");

    BinaryTree oldRight = rightTree;
    oldRight.setParent(null);

    rightTree = cloneEmptyTree();
    rightTree.setParent(this);

    return oldRight;
  }

  //-----------------------------------------------
  // Helpful protected methods (for subclasses).
  //
  // The leftSubtree and rightSubtree methods are protected so that
  // applications cannot have 
  //  leftSubtree and rightSubtree return the Binary Trees that make up the
  //  left and right subtrees, respectively.  As a precondition, this must
  //  non-empty. The functions are protected so that applications cannot have
  //  more than one handle for a single binary tree object.
  //-----------------------------------------------

  protected BinaryTree leftSubtree()
  // pre:  !isEmpty().
  // post: returns the left subtree.
  {
    return leftTree;
  }

  protected BinaryTree rightSubtree()
  // pre:  !isEmpty().
  // post: returns the right subtree.
  {
    return rightTree;
  }

  private void setParent(BinaryTree newParent)
  // pre:  newParent == null or !newParent.isEmpty()
  // post: newParent is the parent of this
  {
    parent = newParent;
  }

  private BinaryTree cloneEmptyTree()
  // Should be overridden in any extension of BinaryTree to get matching type
  // post: returns a new empty tree
  {
    return new BinaryTree();
  }

  protected boolean hasEmptyLeft()
  // pre:  !isEmpty()
  // post: returns true iff the left subtree is empty
  {
    return leftTree.isEmpty();
  }

  protected boolean hasEmptyRight()
  // pre:  !isEmpty()
  // post: returns true iff the right subtree is empty
  {
    return rightTree.isEmpty();
  } 

  protected boolean isNested()
  // post: returns true iff this tree is contained within
  //       some other tree
  {
    return parent != null;
  }

  protected BinaryTreeInterface getParent()
  // post: returns the tree which directly contains this
  //       or null if !isNested()
  {
    return parent;
  }

  protected BinaryTreeInterface rootAncestor()
  // post: returns the root subtree that contains this
  {
    BinaryTree r = this;
 
    while(r.isNested())
      r = (BinaryTree)r.getParent();

    return (BinaryTreeInterface)r;
  }

  //--------------------------------------
  // toString method (for printing a tree) 
  //--------------------------------------

    public String toString() 
    // post: returns String representation of this, using 
    // an inorder traversal with parentheses
    {
      if (this.isEmpty()) return "";
      else return " ("  
            + leftSubtree().toString() 
            + " " 
            + this.getRootItem().toString() 
            + " " 
            + rightSubtree().toString()
            + ") ";
    }


//-----------------------------------------------

  public static void main(String [] args ) {
    BinaryTree a = new BinaryTree(new Integer(1));
    BinaryTree b = new BinaryTree(new Integer(5));
    BinaryTree d = new BinaryTree();

    System.out.println("Empty tree: " + d.isEmpty());
    System.out.println("Printing empty tree:");
    System.out.println(d);
    System.out.println("One item tree: " + a.isEmpty());
    System.out.println("Printing one item tree:");
    System.out.println(a);

    //BinaryTree alpha = new BinaryTree(new Integer(42), a, null);
    BinaryTree beta = new BinaryTree(new Integer(42), a, b);

    System.out.println("Printing typical tree:");
    System.out.println(beta);

    System.out.println("Typical tree: " + beta.isEmpty());
    beta.makeEmpty();
    System.out.println("Cleared Typical tree: " + beta.isEmpty());

    beta.setRootItem(new Integer(1701));
    System.out.println("Printing setRootItem() tree:");
    System.out.println(beta);

    beta.attachLeft(new Integer(2001));
    beta.attachLeft(new Integer(2002));
    beta.attachRight(new Integer(1996));
    beta.attachRight(new Integer(1997));
    System.out.println("Printing attached tree:");
    System.out.println(beta);
    System.out.println();
    System.out.println();
    System.out.println();

    d.setRootItem(new Integer(46));
    d.attachLeftSubtree(beta);
    //d.attachLeftSubtree(beta);
    System.out.println("Printing next tree:");
    System.out.println(d);
    System.out.println();
    System.out.println();
    System.out.println();

    try {
       d.attachRightSubtree(d);
    } catch (TreeException e1) {
        System.out.println("successfully caught attaching " + 
                " tree to itself");
    }
    try {
       d.attachLeftSubtree(beta);
    } catch (TreeException e2) {
        System.out.println("successfully caught double attaching");
    }
    try {
       d.attachRightSubtree(d.leftSubtree());
    } catch (TreeException e2) {
        System.out.println("successfully caught attaching subtree");
    }

    BinaryTree leftResult = d.leftSubtree();
    BinaryTree rightResult = d.rightSubtree();
    BinaryTreeInterface leftSubtree = d.detachLeftSubtree();
    BinaryTreeInterface rightSubtree = d.detachRightSubtree();

    System.out.println("Printing leftSubtree:");
    System.out.println(leftSubtree);
    System.out.println();
    System.out.println();
    System.out.println();
    System.out.println("Printing rightSubtree:");
    System.out.println(rightSubtree);
    System.out.println();
    System.out.println();

  }

} // end BinaryTree
