import math
from contiguous import *

## A class implementing a point using contiguous memory

class Point:
    """
    Field: _data is contiguous memory storing x_value, y_value, and label
    """

    ## Point(x, y, lbl) produces a point with x_value x, y_value y, and
    ##     label lbl.
    ## __init__: Int Int Str -> Point
    def __init__(self, x, y, lbl):
        self._data = Contiguous(3)
        self._data.store(0, x)
        self._data.store(1, y)
        self._data.store(2, lbl)

    ## repr(self) produces a string with the coordinates and label
    ##     of self.
    ## __repr__: Point -> Str
    def __repr__(self):
        return self.label() + ": (" + str(self.x_value()) + \
            ", " + str(self.y_value()) + ")"


    ## self.x_value() produces the x_value of self.
    ## x_value: Point -> Int
    def x_value(self):
        return self._data.access(0)

    ## self.y_value() produces the y_value of self.
    ## y_value: Point -> Int
    def y_value(self):
        return self._data.access(1)

    ## self.label() produces the label of self.
    ## label: Point -> Int
    def label(self):
        return self._data.access(2)

    ## self.same_place(other) produces True if self and other have
    ##     the same x_value and y_value and False otherwise.
    ## same_place: Point Point -> Bool
    def same_place(self, other):
        return self.x_value() == other.x_value() and \
            self.y_value() == other.y_value()

    ## self == other produces True if self and other have the same
    ##     x_value, y_value, and label and False otherwise.
    ## __eq__: Point Point -> Bool
    def __eq__(self, other):
        return self.same_place(other) and \
            self.label() == other.label()

    ## self.dist(other) produces the distance between self and other.
    ## dist: Point Point -> Float
    def dist(self, other):
        x_diff = self.x_value() - other.x_value()
        y_diff = self.y_value() - other.y_value()
        sum_of_squares = pow(x_diff, 2) + pow(y_diff, 2)
        return math.sqrt(sum_of_squares)

    
