#lang racket
; The DFA file used by the program is defined in "dfa.rkt": 
(require "dfa.rkt")

(define STATES      ".STATES")
(define TRANSITIONS ".TRANSITIONS")
(define INPUT       ".INPUT")

; Helper functions

; Read a single string from a port (standard input by default).
; Leading whitespace is skipped until a non-whitespace character is found.
; If a non-whitespace character is found, this function reads characters until
; either whitespace or the end of input is found, and returns the string it read.
; Otherwise (if there is nothing except whitespace before the end of input)
; this function returns eof.
(define (read-chars [in (current-input-port)])
  ; Below is an example of a "named let", which allows you to specify
  ; a one-off function and immediately call it with some default values.
  ; In this case the function is called "helper" and we initially pass
  ; an empty accumulator. The function skips whitespace until it finds
  ; a non-whitespace character, then starts cons-ing characters onto
  ; the accumulator until either more whitespace or EOF is encountered.
  ; This behaves basically the same as the C++ stream extraction operator
  ; (>>) for strings.
  (define result
    (let helper ([accumulator empty])
      (define c (peek-char in))
      (cond [(eof-object? c)
             (cond [(empty? accumulator) eof]
                   [else accumulator])]
            [(char-whitespace? c)
             (cond [(empty? accumulator)
                    (read-char in)
                    (helper accumulator)]
                   [else accumulator])]
            [else (helper (cons (read-char in) accumulator))])))
  ; Because we used cons, the resulting list of characters is actually
  ; in reverse order, so we reverse it before returning the string.
  (cond [(eof-object? result) eof]
        [else (list->string (reverse result))]))

; Check if a string is a range (two characters separated by a - character)
(define (range? s)
  (and (equal? (string-length s) 3) (equal? (string-ref s 1) #\-)))

; Convert a range to the corresponding list of characters.
(define (range->list s)
  (let collect-chars
    [(lo (char->integer (string-ref s 0)))
     (hi (char->integer (string-ref s 2)))]
    (cond [(<= lo hi)
           (cons (integer->char lo) (collect-chars (+ lo 1) hi))]
          [else empty])))

; Print an error message to standard error and exit.
(define (err message)
  (eprintf "ERROR: ~a~n" message)
  (exit))

; Replace all escape sequences in a string with the corresponding characters.
(define (escape str)
  (regexp-replaces
    str
    `(
      [#rx"\\\\s" " "]
      [#rx"\\\\n" "\n"]
      [#rx"\\\\r" "\r"]
      [#rx"\\\\t" "\t"]
      [#rx"\\\\x([0-9A-Fa-f][0-9A-Fa-f])" ,(lambda (all one)
                                              (string (integer->char
                                                      (string->number one 16))))]
      [#px"([[:graph:]])" "\\1"])))

; Replace special characters in a string with escape sequences. 
(define (unescape str)
  (regexp-replaces
    str
    `(
      [#rx" " "\\\\s"]
      [#rx"\n" "\\\\n"]
      [#rx"\r" "\\\\r"]
      [#rx"\t" "\\\\t"]
      [#px"([^[:graph:]])" ,(lambda (all one)
                               (format "\\x~a"
                                       (~r #:base 16
                                           #:min-width 2
                                           #:pad-string "0"
                                           (char->integer (string-ref one 0)))))])))

; DFA printing function

; Print a representation of a DFA file to standard output.
; The argument is a port, so for example, pass (current-input-port)
; to read from standard input, or (open-input-string str) to read
; from the string variable str.
(define (dfa-print input-port)
  ; Skip blank lines at the start of the file
  (let skip-blank ([line (read-line input-port)])
    (when (eof-object? line)
          (err (format "Expected ~a, but found end of input." STATES)))
    (define trimmed (string-trim line))
    (cond [(equal? trimmed STATES) (void)]
          [(non-empty-string? trimmed)
           (err (format "Expected ~a, but found: ~a" STATES trimmed))]
          [else (skip-blank (read-line input-port))]))
  ; Print states
  (displayln "States:")
  (let process-states ([initial #t])
    (define state (read-chars input-port))
    (when (eof-object? state)
      (err (format "Unexpected end of input while reading state set: ~a not found."
                   TRANSITIONS)))
    (cond [(equal? state TRANSITIONS) (void)]
          [else ; Process an individual state
            (define len (string-length state))
            (define last (string-ref state (- len 1)))
            (define accepting (and (equal? last #\!) (> len 1)))
            ; Remove the ! from the name if this is an accepting state
            (define state-name (if accepting
                                 (substring state 0 (- len 1))
                                 state))
            (printf "~a~a~a~n"
                    state-name
                    (if initial   " (initial)"   "")
                    (if accepting " (accepting)" ""))
            ; Recurse on the next element of the states section
            ; Pass #f for initial because we've already seen the initial state
            (process-states #f)]))
  ; Print transitions
  (displayln "Transitions:")
  ; Skip past transitions header
  (read-line input-port)
  (let process-transitions ([line (read-line input-port)])
    (define split (if (eof-object? line) line (string-split line)))
    (cond [(eof-object? split) (void)] ; If we reach the end of file, we're done
          [(equal? (first split) INPUT) (void)] ; If we reach an INPUT section, we're done
          [(empty? split) (process-transitions (read-line input-port))] ; Skip empty lines
          [(< (length split) 3)
           (err (format "Incomplete transition line: ~a" line))]
          [else
            (define from-state (first split))
            (define to-state (last split))
            (define middle (drop-right (rest split) 1))
            ; Print the from-state
            (printf "~a " from-state)
            ; Process and print the characters/ranges
            (let process-middle ([lst middle])
              (define char-or-range (unless (empty? lst) (escape (first lst))))
              (cond [(empty? lst) (void)]
                    [(equal? (string-length char-or-range) 1)
                     ; It's a character
                     (when (> (char->integer (string-ref char-or-range 0)) 127)
                       (err "Invalid (non-ASCII) character in transition line: ~a~nCharacter ~a is outside ASCII range"
                            line
                            (unescape char-or-range)))
                     (printf "~a " (unescape char-or-range))
                     (process-middle (rest lst))]
                    [(range? char-or-range)
                     (map
                      (lambda (c)
                        (printf "~a " (unescape (string c))))
                      (range->list char-or-range))
                     (process-middle (rest lst))]
                    [else (err (format "Expected character or range, but found ~a in transition line: ~a"
                                       char-or-range
                                       line))]))
            ; Print the to-state
            (printf "~a~n" to-state)
            ; Recurse on the next line
            (process-transitions (read-line input-port))])))


; By default, print the DFA defined in the DFA-STRING constant provided by "dfa.rkt"
(dfa-print (open-input-string DFA-STRING))
